<?php

namespace App\Http\Controllers\admin;

use App\Http\Controllers\Controller;
use App\Models\Notification;
use App\Models\OrderItems;
use App\Models\OrderStatusHistory;
use GrahamCampbell\ResultType\Success;
use Illuminate\Http\Request;
use App\Models\Order;
use App\Models\OrderItemHistory;
use DB;

class OrderController extends Controller
{

    public function list(){
        $orders = Order::filter()->orderBy('id','desc')->get();
        return view('admin.order.list',compact('orders'));
    }

    public function view($id){
        $order = Order::where('id',$id)->first();
        return view('admin.order.view',compact('order'));
    }

    public function changeItemStatus(Request $request,$order_item_id){
        DB::beginTransaction();
        try{
            $request->validate([
                'status'=>'required',
                'note'=>'required'
            ]);
            $orderItem = OrderItems::where('id',$order_item_id)->first();
            if(!$orderItem){
                abort(404);
            }
            $orderItem->update(['status'=>$request->status]);
            $data = [
                'order_id'=>$orderItem->order_id,
                'order_item_id'=>$orderItem->id,
                'product_id'=>$orderItem->product_id,
                'status'=>$request->status,
                'note'=>$request->note,
                'hub_id'=>$orderItem->hub_id
            ];


            OrderItemHistory::updateOrCreate(['order_item_id'=>$orderItem->id],$data);
            DB::commit();
            flash()->success('Order Item Status Changed Successfully');
            return redirect()->back();
        }catch(\Exception $e){
            return $e;
        }
    }

    public function changeOrderStatus($order_id,$status){
        DB::beginTransaction();
        try{
            $order = Order::with('user')->findOrFail($order_id);

            if($status == ORDER_REACHED_IN_DELIVERY_HUB){
                $orderItems = OrderItems::where('order_id',$order_id)->where('status',REACHED_AT_LOCAL_HUB)->first();
                if($orderItems){
                    flash()->warning("Not all items reached in delivery hub");
                    return redirect()->back();
                }
            }

            $order->update(['status'=>$status]);
            $by = auth('user')->check() && auth('user')->user()->role == ADMIN?'admin':'';
            $statusData =  [
                'order_id'=>$order->id,
                'status_id'=>$status,
                'note'=>'Order status has been changed by '.$by
            ];
            OrderStatusHistory::updateOrCreate(['order_id'=>$order->id,'status_id'=>$status],$statusData);

            if($status == ORDER_DELIVERED && !$order->is_comission_shared){
                if($order->user?->parent_id){
                    Order::distributeReferelBonus($order->user_id,$order->total_spot_comission,$order->total_team_comission,$order_id);
                }
                $order->update(['is_comission_shared'=>1]);
            }

            if($status == ORDER_DELIVERED){
                OrderItems::with('product')->where('order_id',$order_id)->get()->map(function($orderItem){
                    $refundEndTime = date('Y-m-d H:i:s', strtotime(date('Y-m-d H:i:s') . '+'.$orderItem?->product?->refund_time.' days'));
                    $orderItem->update(['refund_end_time'=>$refundEndTime]);
                    $orderItem->product?->increment('total_sold',$orderItem->qty);
                });
            }


            $this->sendNotification($order->order_number,$order->user_id,$order->phone,$order->status);


            DB::commit();
            flash()->success("Order status changed successfully");
            return redirect()->back();
        }catch(\Exception $e){
            return $e;
        }
    }

    public function sendNotification($userId,$orderNumber,$phone,$status){
        $orderStatus = [
            ORDER_ACCEPTED => "Welcome to ezone . We accpted your order(#".$orderNumber."). We will notify you in every steps of your order process",
            ORDER_READY_FOR_DELIVERY => "Your order(#".$orderNumber.") is ready for delivery, out driver will reach the items to you very soon",
            ORDER_OUT_FOR_DELIVERY => "Your order(#".$orderNumber.") is out for delivery . our handled your items to our delivery man , he will reach to you very soon",
            ORDER_REACHED_IN_DELIVERY_HUB => "Your order(#".$orderNumber.") items reached in our delivery hub . very soon we will deliver it to you",
            ORDER_DELIVERED => "We deliverd your order(#".$orderNumber.") . please check the order items , if you found broken/unordered/unexpected item please contact with us ASAP",
        ];
        $message = $orderStatus[$status].' https://admin.robiulawal.xyz/';
        $title = getStatus($status);
        Notification::addNotification($title,$message,$userId);
        Notification::sendSms($message,$phone);
        return true;
    }


    public function testNotification(){
        
    }
}
