<?php
namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class Product extends Model
{
    use SoftDeletes;
    public $primaryKey = "id";
    public $guarded = [];

    protected $appends = ["thumbnail_url","actual_sell_price","actual_sell_price_format"];
    public function getThumbnailUrlAttribute()
    {
        if (optional($this->thumbnailImage)->file_path) {
            return url("images/" . $this->thumbnailImage->file_path);
        } else {
            return noImage();
        }
    }

    public function scopeFilter($query)
    {
        $status = request("status");
        $type=request('_type');

        $brandId = request('brand_id');
        $categoryId = request('category_id');
        $subcategoryId = request('subcategory_id');
        $companyId = request('company_id');
        $shopId = request('shop_id');
        $status = request('status');

        if($brandId){
            $query->where('brand_id',$brandId);
        }

        if($categoryId){
            $query->where('category_id',$categoryId);
        }

        if($companyId){
            $query->where('company_id',$companyId);
        }

        if($subcategoryId){
            $query->where('subcategory_id',$subcategoryId);
        }

        if($status>-1){
            $query->where('status',$status);
        }

        if($shopId){
            $query->where('shop_id',$shopId);
        }


        if($type){
            if($type=='seller'){
                $query->whereHas('vendor',function($query){
                    $query->where('type',SELLER);
                });
            }
            if($type=='deller'){
                $query->whereHas('vendor',function($query){
                    $query->where('type',DELLER);
                });
            }
        }

        $query->whereNull('deleted_at');
    }

    public function scopeFilterdata($query)
    {
        $isFeatured = request('is_featured');
        $bestSeller = request('best_seller');
        $latestProduct = request('latest_product');
        $category = request('category_id');
        $subcategory = request('subcategory_id');
        $shop = request('shop_id');
        $basicInfoId = request('basic_info_id');
        $groupId = request('group_id');
        $search = request('search');
        $productType = request('product_type');
        $type = request('type');

        if($groupId){
            $groupProducts = GroupProduct::where('id',$groupId)->first();
            $query->whereIn('id',$groupProducts->product_ids);
        }

        if($search){
            $query->whereHas('basicInfo', function($query)use($search){
                $query->where('product_name_en','LIKE','%'.$search.'%')
                    ->orwhere('product_name_bn','LIKE','%'.$search.'%');
            });
        }



        if($isFeatured){
            $query->where('is_featured',1);
        }

        if($category){
            $query->where('category_id',$category);
        }

        if($subcategory){
            $query->where('subcategory_id',$subcategory);
        }

        if($shop){
            $query->where('shop_id',$shop);
        }

        if($basicInfoId){
            $query->where('basic_info_id',$basicInfoId);
        }

        if($type){
            $query->where('type',$type);
        }

        if($productType){
            $query->where('product_type',$productType);
        }else{
            //1 means customer product
            $query->where('product_type',0);
        }

        if($bestSeller){
            $query->orderBy('total_sold','desc');
        }

        if($latestProduct){
            $query->orderBy('id','desc');
        }


        $query->where("status", ACTIVE)->whereNull('deleted_at');
        // $query;
    }
    public function basicInfo():BelongsTo
    {
        return $this->belongsTo(ProductBasicInfo::class, "basic_info_id", "id");
    }

    public function shop():BelongsTo
    {
        return $this->belongsTo(Shop::class, "shop_id", "id");
    }

    public function vendor():BelongsTo
    {
        return $this->belongsTo(Vendor::class, "vendor_id", "id");
    }

    public function brand():BelongsTo
    {
        return $this->belongsTo(Brands::class, "brand_id", "id");
    }

    public function category():BelongsTo
    {
        return $this->belongsTo(ProductCategory::class, "category_id", "id");
    }

    public function subcategory():BelongsTo
    {
        return $this->belongsTo(ProductSubcategory::class, "subcategory_id", "id");

    }

    public function company():BelongsTo
    {
        return $this->belongsTo(Company::class,'company_id');
    }

    public function images(): HasMany
    {
        return $this->hasMany(ProductImages::class,'product_id');
    }

    public function colors(): HasMany
    {
        return $this->hasMany(ProductColor::class,'product_id','id');
    }

    public function getActualSellPriceAttribute(){
        if($this->customer_discount){
            $sellPrice = (float)($this->sale_price-($this->sale_price*$this->customer_discount/100));
            return $sellPrice;
        }

        return $this->sale_price;
    }

    public function getActualSellPriceFormatAttribute(){
        if($this->customer_discount){
            return number_format($this->sale_price - $this->sale_price*$this->customer_discount/100,2);
        }

        return $this->sale_price;
    }

    public function thumbnailImage(): BelongsTo
    {
        return $this->belongsTo(Images::class,'thumbnail');
    }


    public function getNameAttribute(): string
    {
        if(getLang()=='bn'){
            return $this->product_name_bn;
        }else{
            return $this->product_name_en;
        }
    }
}
