<?php

namespace App\Http\Controllers\api;

use App\Http\Controllers\Controller;
use App\Http\Resources\Api\BannerResource;
use App\Http\Resources\Api\CategoryProductResource;
use App\Http\Resources\Api\CategoryResource;
use App\Http\Resources\Api\GroupResource;
use App\Http\Resources\Api\ProductCardInfoResource;
use App\Http\Resources\Api\ShopResource;
use App\Http\Resources\Api\ProductDetialsResource;
use App\Models\Banner;
use App\Models\Brands;
use App\Models\Category;
use App\Models\CategoryImages;
use App\Models\GroupProduct;
use App\Models\Offer;
use App\Models\Order;
use App\Models\Product;
use App\Models\ProductBasicInfo;
use App\Models\ProductCategory;
use App\Models\ProductHideTime;
use App\Models\ProductSubcategory;
use App\Models\Shop;
use App\Models\Notification;
use App\Models\OfferProducts;
use Carbon\Carbon;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class ProductControler extends Controller
{
    public function getSubCategories(): JsonResponse
    {
        $data = ProductSubcategory::select('id','subcategory_name_en')->filter()->whereHas('products')->get();
        return response()->json(['data'=>$data]);
    }

    public function getHomePageData(): JsonResponse
    {
        try {
            $allBanner = Banner::where(function($query){
                $query->where('publish_at','<=',Carbon::now())->where('unpublish_at','>=',Carbon::now());
            })->orWhereNull('unpublish_at')->whereNotNull('banner_image')->where('status', ACTIVE)->get();

            $banners = $allBanner->where('id','>',0);
            $popupBanner = $allBanner->where('id',0)->first();

            $featuredProducts = Product::with('thumbnailImage')->withWhereHas('basicInfo')
                    ->where('is_featured', ACTIVE)
                    ->where('show_in_home', SHOW_IN_HOME)
                    ->where('status', ACTIVE)
                    ->whereNull('deleted_at')
                    ->take(6)
                    ->get();

            $latestProducts = Product::with('thumbnailImage')->withWhereHas('basicInfo')
                    ->where('show_in_home', SHOW_IN_HOME)
                    ->where('status', ACTIVE)
                    ->orderBy('created_at', 'desc')
                    ->whereNull('deleted_at')
                    ->take(10)
                    ->get();

            $categories = ProductCategory::whereHas('products')
                    ->where('status', ACTIVE)
                    ->where('show_in_home', SHOW_IN_HOME)
                    ->take(10)
                    ->get();

            $shops = Shop::whereHas('products')
                    ->where('status', ACTIVE)
                    ->take(10)
                    ->get();

            $bestSellers = Product::with('thumbnailImage')->withWhereHas('basicInfo')
                    ->where('status',ACTIVE)
                    ->orderBy('total_sold','DESC')
                    ->whereNull('deleted_at')
                    ->take(6)
                    ->get();

            $today = now();
            $groups = GroupProduct::where('status',1)->get();
            // $offerBanner = Offer::where('start_date', '<=', $today)->where('end_date', '>=', $today)->get();
            // $unread_total_notification = 0;
            // if(auth()->check())
                // $unread_total_notification = Notification::where('id',auth()->user()->id)->where('is_read',0)->count();

            return response()->json([
                'banner' => BannerResource::collection($banners),
                'popupBanner'=>$popupBanner?new BannerResource($popupBanner):null,
                'featuredProducts' => ProductCardInfoResource::collection($featuredProducts),
                'latestProducts' => ProductCardInfoResource::collection($latestProducts),
                'categories' => CategoryResource::collection($categories),
                'shops' => ShopResource::collection($shops),
                'bestSellers'=>ProductCardInfoResource::collection($bestSellers),
                // 'unread_total_notification'=>$unread_total_notification,
                'is_logged_in'=>auth('sanctum')->check()?true:false,
                'groups'=>GroupResource::collection($groups)
                // 'offerBanner' => $offerBanner,
            ], 200);
        } catch (\Exception $e) {
            // Log or handle the exception appropriately
            return response()->json([
                'message' => 'Something went wrong',
                'err' => $e,
            ], 500);
        }
    }

    public function getCategories(): JsonResponse
    {
        $categories = ProductCategory::whereHas('products')->where('status', ACTIVE)->get();
        return response()->json(['success'=>true,'data'=>CategoryResource::collection($categories)]);
    }

    public function getShops(): JsonResponse
    {
        $shops = Shop::whereHas('products')->where('status', ACTIVE)->get();
        return response()->json(['success'=>true,'data'=>ShopResource::collection($shops)]);
    }

    public function getCategoryProducts(): JsonResponse
    {
        $categories = ProductCategory::withWhereHas('products',function($query){
            $query->whereHas('basicInfo')->take(10);
        })->where('show_in_home',1)->take(5)->get();
        return response()->json(['success'=>true,'data'=>CategoryProductResource::collection($categories)]);
    }

    public function getProductInfo(): JsonResponse
    {
        $categoryBanners =  [];
        $title = 'Explore product';
        $subtitle = 'Please explore all of our products';
        if(request('category_id')){
            $title = ProductCategory::where('id',request('category_id'))->value('category_name_'.getLang());
            $subtitle = 'You are now viewing '.strtolower($title).' products';
            $categoryBanners = CategoryImages::where('category_id',request('category_id'))->get();
            $categoryBanners = $categoryBanners->pluck('image_url');
        }

        if(request('group_id')){
            $title = GroupProduct::where('id',request('group_id'))->first()?->name;
            $subtitle = 'You are now viewing '.strtolower($title).' products';
        }

        if(request('shop_id')){
            $title = Shop::where('id',request('shop_id'))->value('shop_name_'.getLang());
            $subtitle = 'You are now viewing '.strtolower($title).' products';
        }

        if(request('is_featured')){
            $title = 'Fatured Products';
            $subtitle = 'You are now viewing Featured Products';
        }

        if(request('best_seller')){
            $title = 'Top Selling Products';
            $subtitle = 'You are now viewing Top Selling Products';
        }

        if(request('latest_product')){
            $title = 'Latest Products';
            $subtitle = 'You are viewing our Latest Products';
        }

        if(request('search')){
            $title = 'Searched Products';
            $subtitle = 'Viewing search results for : '.request('search');
        }

        return response()->json(['categoryBanners'=>$categoryBanners,'title'=>$title,'subtitle'=>$subtitle]);
    }

    public function getProductInGroup ()
    {
        try{
            $data = Product::withWhereHas('basicInfo')->filterdata()->paginate(10);
            return response()->json([
                'data'=>ProductCardInfoResource::collection($data->items()),
                'total_page'=>$data->lastPage(),
                'current_page'=>$data->currentPage(),
                'types'=>productType(),
            ]);
        }catch(\Exception $e) {
            return response()->json([
                'message' => 'Internal Server Error',
            ], 500);
        }
    }

    public function getOfferProduts($offerId){
        $offer = Offer::select('id','image','offer_name_'.getLang().' as offer_name','amount_type','amount')
        ->where('id',$offerId)
        ->whereDate('start_date','<=',Carbon::now())
        ->whereDate('end_date','>=',Carbon::now())
        ->first();
        if($offer){
            $offerProducts = OfferProducts::where('tbl_offer_products.id',$offerId)
            ->selectRaw(
                'tbl_product.id,
                thumbnail,
                product_name_'.getLang().' as product_name,
                sale_price,
                original_price'
            )
            ->leftjoin('tbl_product','tbl_offer_products.id','=','tbl_product.id')
            ->paginate(20);

            if($offer->amount_type == 'F'){
                foreach($offerProducts as $offerProduct)
                    $offerProduct->actual_sell_price = (float)number_format($offerProduct->sale_price - $offer->amount,2);
                    $offerProduct->actual_sell_price_format = number_format($offerProduct->sale_price - $offer->amount,2);
            }else{
                foreach($offerProducts as $offerProduct)
                    $offerProduct->actual_sell_price = (float)number_format($offerProduct->sale_price - $offerProduct->sale_price*$offer->amount/100,2);
                    $offerProduct->actual_sell_price_format = number_format($offerProduct->sale_price - $offerProduct->sale_price*$offer->amount/100,2);
            }

            return response()->json(['offer'=>$offer,'offer_products'=>$offerProducts->items(),'total_page'=>$offerProducts->total(),'current_page'=>$offerProducts->currentPage()]);
        }

        return response()->json(['offer'=>$offer,'offer_products'=>[],]);
    }


    public function getProductDetails($id) {
        try{
            $product = Product::with(['images.image','shop','brand','category','subcategory','colors'])->where('id',$id)->first();
            $relatedProduct = Product::where('id','!=',$id)->where('subcategory_id', $product->subcategory_id)->where('status',ACTIVE)->take(4)->get();
            return response()->json([
                'data' => new ProductDetialsResource($product),
                'relatedProduct' => ProductCardInfoResource::collection($relatedProduct),
                'is_logged_in'=>auth('sanctum')->check()?true:false,
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Internal Server Error',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function productBasicInfo(){
        $products = ProductBasicInfo::select('id','product_name_en','product_name_bn')->get();
        return response()->json(['success'=>true,'products'=>$products]);
    }

    public function getProductBasicInfo(){
        $products = ProductBasicInfo::select('id','product_name_en','product_name_bn')
                ->where('status',1)
                ->with(['thumbnails','images'])
                ->filter()
                ->get();
        return response()->json(['success'=>true,'products'=>$products]);
    }

    public function getSubcategoryByCategory($categoryId){
        $data = ProductSubcategory::select('id','subcategory_name_en')->where('category_id',$categoryId)->where('status',ACTIVE)->get();
        return response()->json(['data'=>$data]);
    }

    public function getBrands($companyId){
        $brands = Brands::where(['company_id'=>$companyId])->get();
        return response()->json(['success'=>true,'data'=>$brands]);
    }
}
