<?php

namespace App\Http\Controllers\api;

use App\Http\Controllers\Controller;
use App\Http\Requests\Api\CouponRequest;
use App\Http\Requests\Api\OrderRequest;
use App\Http\Requests\Api\ReturnRequest;
use App\Http\Resources\Api\OrderItemResource;
use App\Models\AppSettings;
use App\Models\CashWalletHistory;
use App\Models\Charges;
use App\Models\ComissionWalletHistory;
use App\Models\Coupon;
use App\Models\Images;
use App\Models\Notification;
use App\Models\Order;
use App\Models\OrderItemHistory;
use App\Models\OrderItems;
use App\Models\OrderStatusHistory;
use App\Models\OrderTransaction;
use App\Models\Product;
use App\Models\ProductImages;
use App\Models\ReturnedItem;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use App\Http\Resources\ReturnItemsResource;
use App\Models\OrderAddress;
use App\Models\Shop;
use App\Models\StockistArea;
use App\Models\UserAddress;
use stdClass;

class OrderControler extends Controller
{
    public function placeOrder (OrderRequest $request) {
        DB::beginTransaction();
        try{
            $user = auth()->user();
            $coupon = null;
            $orderMax = date('ymdhis').rand(111,999);
            $orderNumber = $orderMax + 1;
            $items = $request->items;
            $qty=0;
            $itemTotal = 0;
            $discountAmount = 0;
            $shipping_fee = 0;
            $spotComission = 0;
            $teamComission = 0;
            $tax = (float)AppSettings::where('tag','order_tax')->value('value');
            $notAvailable = [];
            $totalDeliveryFee = 0;
            $isDeliveryInsideDhaka = false;
            $hubId = null;


            if(in_array(auth()->user()->division_id,DELIVERY_INSIDE_DHAKA)){
                $isDeliveryInsideDhaka = true;
            }

            $shippingAddress = UserAddress::where('id',$request->addressId)->first();
            if(!$shippingAddress){
                return response()->json(['success'=>false,'message'=>'Invalid shipping address'],400);
            }
            // TO DO: if address type is mystore then its a my store order so asign this order to a stockist
            if($shippingAddress->type == 'mystore'){
                $stockistArea = StockistArea::where('area_id',$shippingAddress->area_id)->first();
                if(!$stockistArea){
                    return response()->json(['success'=>false,'message'=>'Invalid shipping address'],400);
                }
                $hubId = $stockistArea->hub_id;
            }

            $cartItems = [];

            foreach ($items as $key=>$_shop) {
                $shop = Shop::where('id',$_shop['store_id'])->where('status',1)->first();
                $isDeliveryFree = false;
                foreach($_shop['items'] as $item){
                    $product = Product::where('id',$item['product_id'])->where('status',ACTIVE)->where('stock','>=',$item['quantity'])->first();
                    if($product){
                        if($product->is_free_delivery){
                            $isDeliveryFree = true;
                        }
                        $itemTotal += $product->actual_sell_price*$item['quantity'];
                        $qty += $item['quantity'];
                        $item['sell_price']=$product->actual_sell_price;
                        $item['shop_id']=$shop->id;
                        $item['hub_id']=$shop->hub_id;
                        $spotComission += (float)$product->spot_comission*$qty;
                        $teamComission += (float)$product->team_comission*$qty;
                        $cartItems[] = $item;
                        $product->decrement('stock',$item['quantity']);
                    }else{
                        $notAvailable[] = $item['product_name'];
                    }
                }
                if(!$isDeliveryFree){
                    if($isDeliveryInsideDhaka){
                        $totalDeliveryFee += $shop->delivery_charge_inside_dhaka;
                    }else{
                        $totalDeliveryFee += $shop->delivery_charge_outside_dhaka;
                    }
                }
            }


            if(count($notAvailable) > 0){
                return response()->json(['success'=>false,'message'=>implode(',',$notAvailable).' -- Sorry this product is not avilable'],409);
            }


            if($request->couponCode){
                $today = now();
                $coupon = Coupon::where('start_date', '<=', $today)
                        ->where('end_date', '>=', $today)
                        ->where('coupon_code', $request->couponCode)
                        ->first();
                if(!$coupon){
                    return response()->json(['success'=>false,'message'=>'The token you used it is invalid or expired'],400);
                }

                if($coupon->minimum_order_amount < $itemTotal){
                    return response()->json(['success'=>false,'message'=>'The token you used it is invalid or expired'],400);
                }

                if($coupon->amount_type == 'F'){
                    $discountAmount = $coupon->amount;
                }else if($coupon->amount_type == 'P'){
                    $discountAmount = $itemTotal * ($coupon->amount / 100);
                }
            }

            $orderTotal = (float)($itemTotal+$tax+$totalDeliveryFee)-$discountAmount;
            $paymentMethod = '';
            $collectedFromWallet = 0;
            $commissionWalletHistoryData = null;
            $cashWalletHistoryData = null;

            $user = User::select('id','cash_wallet','comission_wallet')->where('id',Auth::user()->id)->first();
            if($request->isWalletUsed){
                $paymentMethod = 'W';

                if((float)$user->comission_wallet >= (float)$orderTotal){
                    $collectedFromWallet += (float)$orderTotal;
                    $user->decrement('comission_wallet',(float)$orderTotal);
                    $commissionWalletHistoryData = $this->getComissionWalletHistoryData($orderTotal);
                }else{
                    $collectedFromWallet += (float)$user->comission_wallet;
                    $user->update(['comission_wallet'=>0]);
                    $commissionWalletHistoryData = $this->getComissionWalletHistoryData($user->comission_wallet);
                }

                if($collectedFromWallet < $orderTotal){
                    $due = (float)$orderTotal-$collectedFromWallet;
                    $paymentMethod = 'W';
                    if((float)$user->cash_wallet >= (float)$due){
                        $collectedFromWallet += (float)$due;
                        $user->decrement('cash_wallet',(float)$due);
                        $cashWalletHistoryData = $this->getCashWalletHistoryData($due);
                    }else{
                        $collectedFromWallet += (float)$user->cash_wallet;
                        $user->update(['cash_wallet'=>0]);
                        $cashWalletHistoryData = $this->getCashWalletHistoryData($user->cash_wallet);
                    }
                }
            }


            if($collectedFromWallet < $orderTotal){
                if($request->paymentMethod){
                    $paymentMethod = $request->paymentMethod;
                }else{
                    $paymentMethod = 'COD';
                }
            }


            //  Insert Order in Order table
            $order = Order::create([
                'order_number' => $orderNumber,
                'user_id' => $user->id,
                'dropshipper_id' => $request->dropshipperId,
                'qty' => $qty,
                'payment_method' => $paymentMethod,
                'order_total' => $itemTotal,
                'tax' => $tax,
                'delivery_charge' => $totalDeliveryFee,
                'status' => ORDER_PENDING,
                'payment_status' => 1,
                'hub_id' => $hubId,//this will be used for stockist
                'coupon_id' => $coupon?$coupon->id:null,
                'discount_amount' => $discountAmount,
                'paid_from_wallet' => $collectedFromWallet,
                'due' => $orderTotal-$collectedFromWallet,
                'order_from' => 'customer',
                'total_spot_comission'=>$spotComission,
                'total_team_comission'=>$teamComission,
                'order_type'=>$request->type
            ]);


            OrderAddress::create([
                'order_id'=>$order->id,
                'phone'=>$shippingAddress->phone,
                'email'=>$shippingAddress->email,
                'division_id'=>$shippingAddress->division_id,
                'district_id'=>$shippingAddress->district_id,
                'upazila_id'=>$shippingAddress->upazila_id,
                'address'=>$shippingAddress->address,
                'area_id'=>$shippingAddress->area_id,
                'user_id'=>auth()->user()->id,
            ]);

            $isDeliveryFree = false;

            //  Insert Item in items table
            foreach ($cartItems as $item) {
                $orderItem =  OrderItems::create([
                    'order_id' => $order->id,
                    'product_id' => $item["product_id"],
                    'shop_id' => $item["shop_id"],
                    'qty' => $item["quantity"],
                    'price' => $item['sell_price'],
                    'discount' => $item['customer_discount'],
                    'status' => 1,
                    'hub_id' => $item['hub_id'],
                    'size' => $item['size'],
                    'color_id' => (int)$item['color_id'] > 0?$item['color_id']:null,
                ]);

                OrderItemHistory::create([
                    'order_id' => $order->id,
                    'order_item_id'=>$orderItem->id,
                    'product_id' => $item["product_id"],
                    'hub_id' => $item['hub_id'],
                    'note'=>'Seller recived the order :)',
                    'status'=>NOT_REACHED_AT_HUB,
                ]);
            }

            OrderTransaction::create([
                'order_id' => $order->id,
                'user_id' => $user->id,
                'payment_method' => $request->paymentMethod,
                'transaction_id' => rand(123456789,987654321),
            ]);

            OrderStatusHistory::create([
                'order_id' => $order->id,
                'status_id' => 1,
                'note' => 'Order place by user',
            ]);




            if($cashWalletHistoryData){
                $cashWalletHistoryData['order_id']=$order->id;
                CashWalletHistory::create($cashWalletHistoryData);
            }

             if($commissionWalletHistoryData){
                $commissionWalletHistoryData['order_id']=$order->id;
                ComissionWalletHistory::create($commissionWalletHistoryData);
            }

            Notification::sendNotification('Order Placed','You place a new order',auth()->user()->id);


            DB::commit();

            return response()->json([
                "sucess"=>true,
                "message" => "Thanks for your order 😍",
                "orderId"=>$order->id,
            ], 200);

        }catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'message' => $e->getMessage(),
                'error' => $e->getMessage()
            ], 500);
        }
    }


    public function getComissionWalletHistoryData($orderTotal){
        return [
            'user_id'=>Auth::user()->id,
            'transaction_type'=>2,
            'amount'=>-$orderTotal,
            'message'=>$orderTotal.'৳ has been deducted for order',
            'for_admin'=>0,
            'current_balance'=>User::where('id',Auth::user()->id)->value('comission_wallet')
        ];
    }

    public function getCashWalletHistoryData($orderTotal){
        return [
            'user_id'=>Auth::user()->id,
            'transaction_type'=>2,
            'amount'=>-$orderTotal,
            'message'=>$orderTotal.'৳ has been deducted for order',
            'payment_method'=>'wallet',
            'current_balance'=>User::where('id',Auth::user()->id)->value('cash_wallet'),
            'status'=>1
        ];
    }


    public function getOrderListForUser () {
        try{
            $user = auth()->user();
            $userId = $user->id;
            $orders = Order::select('id', 'order_number', 'status', 'created_at')->where('user_id', $userId)->orderBy('id','desc')->get();

            return response()->json([
                "message" => "User Orders Sent.",
                "orders" => $orders,
            ], 200);

        }catch (\Exception $e) {
            return response()->json([
                'message' => "Internal Server Error",
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function getOrderDetails ($id) {
        try{
            $user = auth()->user();
            $data = Order::with(['orderItems.product.thumbnailImage','orderItems.color','address','address.division','address.district','address.upazila'])
            ->select('id', 'order_number', 'status', 'order_total','tax','delivery_charge','created_at','discount_amount','paid_from_wallet','due','updated_at')
            ->where('id', $id)
            ->where('user_id', auth()->id())
            ->first();

            $orderItems = collect([]);

            $groupedOrderItems = $data->orderItems?->groupBy('shop_id');
            // foreach()
            foreach($groupedOrderItems as $shopId=>$items){
                $info = new stdClass();
                $shop = Shop::where('id',$shopId)->first();
                if($shop){
                    $info->store_id = $shop->id;
                    $info->store_name = $shop->name;
                    $info->items = OrderItemResource::collection($items);
                }

                $orderItems->push($info);
            }




            $statusTimes = OrderStatusHistory::where('order_id', $id)
            ->selectRaw('status_id, DATE_FORMAT(created_at, "%d %b %Y %H:%i") as formatted_created_at')
            ->pluck('formatted_created_at', 'status_id');

            return response()->json([
                "message" => "Order details Sent.",
                "data" => $data,
                "statusTimes" => $statusTimes,
            ], 200);

        }catch (\Exception $e) {
            return response()->json([
                'message' => "Internal Server Error",
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function returnItem (ReturnRequest $request) {
        DB::beginTransaction();
        try{
            $inactiveStatus = notRefundableStatus();
            $orderItem = OrderItems::where('id',$request->orderItemId)->first();

            if(!$orderItem){
                return response()->json([
                    "message" => "Sorry we didn't found this item",
                ], 200);
            }

            if(in_array($orderItem->status,$inactiveStatus)){
                return response()->json([
                    "message" => "Sorry you already made a return request",
                ], 400);
            }


            $orderItem->update(['status'=>RETURN_REQUEST]);


            ReturnedItem::updateOrCreate(['order_item_id'=>$orderItem->id],[
                'order_id' => $orderItem->order_id,
                'user_id' => auth()->user()->id,
                'order_item_id' => $orderItem->id,
                'product_id'=>$orderItem->product_id,
                'shop_id'=>$orderItem->shop_id,
                'reason' => $request->reason,
                'note' => $request->note,
            ]);

            OrderItemHistory::create([
                'order_id'=> $orderItem->order_id,
                'order_item_id'=> $orderItem->id,
                'product_id'=> $orderItem->product_id,
                'hub_id'=>1,
                'note'=>$request->note,
                'status'=>RETURN_REQUEST
            ]);

            DB::commit();

            return response()->json([
                "message" => "Return request successfull.",
            ], 200);

        }catch (\Exception $e) {
            return response()->json([
                'message' => "Internal Server Error",
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function returnRequests () {
        try{
            $returnItems = ReturnedItem::with(['product','orderItem.shop.user','orderItem.product'])->where('user_id',auth()->user()->id)->orderBy('id','desc')->get();
            return response()->json([
                "message" => "Return Requst data Sent.",
                "data"=>ReturnItemsResource::collection($returnItems)
            ], 200);

        }catch (\Exception $e) {
            return response()->json([
                'message' => "Internal Server Error",
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function getCharges(){
        $appSettings = AppSettings::pluck('value', 'tag');
        $userInfo = null;
        if(auth('sanctum')->check()){
            $userInfo = User::select('id','first_name','last_name','phone','email','address','district_id','division_id','upazila_id','cash_wallet','comission_wallet')
            ->where('id',auth('sanctum')->user()->id)
            ->with(['division','district','upazila'])
            ->first();
        }

        return response()->json(['success'=>true,'charges'=>$appSettings,'userInfo'=>$userInfo]);
    }


    public function applyCoupon(CouponRequest $request){
        try{
            $today = Carbon::now();
            $coupon = Coupon::where('start_date', '<=', $today)->where('end_date', '>=', $today)->where('coupon_code',$request->coupon_code)->first();
            if(!$coupon){
              return response()->json(['success'=>false,'message'=>'Invalid coupon code'],400);
            }

            $total = $request->total;
            $discount = 0;
            if($coupon->amount_type == 'P'){
                $discount = $total*$coupon->amount/100;
            }else{
                $discount = $coupon->amount;
            }

            return response()->json(['success'=>true,'discount_amount'=>$discount],200);
        }catch(\Exception $e){
            return $e;
        }
    }

    public function cancelOrder($orderId){
        DB::beginTransaction();
        $order = Order::where('id',$orderId)->where('status',ORDER_PENDING)->first();
        if(!$order)
            return response()->json(['status'=>false,'message'=>"You can't cancel this order"],500);
        $order->update(['status'=>ORDER_CANCELLED]);
        OrderStatusHistory::create([
            'order_id' => $order->id,
            'status_id' => ORDER_CANCELLED,
            'note' => 'Order cancelled by user',
        ]);
        DB::commit();
        return response()->json(['status'=>false,'message'=>"Order Cancelled Successfully"],200);
    }
}
